<?php

namespace App\Http\Controllers\Auth;

use App\Enums\TxnStatus;
use App\Enums\TxnType;
use App\Events\UserReferred;
use App\Http\Controllers\Controller;
use App\Models\LoginActivities;
use App\Models\Page;
use App\Models\Ranking;
use App\Models\User;
use App\Providers\RouteServiceProvider;
use App\Rules\Recaptcha;
use App\Traits\NotifyTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Log;

class RegisteredUserController extends Controller
{
    use NotifyTrait;

    /**
     * Display the registration view with math captcha.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        if (!setting('account_creation', 'permission')) {
            abort(403, 'User registration is closed now');
        }

        $page = Page::where('code', 'registration')->where('locale', app()->getLocale())->first();
        $data = json_decode($page->data, true);

        $googleReCaptcha = plugin_active('Google reCaptcha');
        $location = getLocation();

        // Generate math captcha numbers and store answer in session
        $num1 = rand(1, 10);
        $num2 = rand(1, 10);
        $sum = $num1 + $num2;
        session(['math_answer' => $sum]);

        Log::info('Math answer set in session: ' . session('math_answer'));

        return view('frontend::auth.register', compact('location', 'googleReCaptcha', 'data', 'num1', 'num2'));
    }

    /**
     * Handle an incoming registration request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        // Honeypot trap for bots
        if ($request->filled('hp_email')) {
            Log::warning('Honeypot caught bot from IP: ' . $request->ip());
            abort(403, 'Bot activity detected.');
        }

        // Log input and session answer for debugging
        Log::info('User entered math answer: ' . $request->input('math_captcha'));
        Log::info('Session math answer: ' . session('math_answer'));

        // Check math captcha answer
        if ($request->input('math_captcha') != session('math_answer')) {
            return back()->withErrors(['math_captcha' => 'Incorrect answer to the math question.'])->withInput();
        }

        $isUsername = (bool) getPageSetting('username_show');
        $isCountry = (bool) getPageSetting('country_show');
        $isPhone = (bool) getPageSetting('phone_show');

        // Validate all inputs including reCAPTCHA if enabled
        $request->validate([
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'username' => [Rule::requiredIf($isUsername), 'string', 'max:255', 'unique:users'],
            'country' => [Rule::requiredIf($isCountry), 'string', 'max:255'],
            'phone' => [Rule::requiredIf($isPhone), 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'g-recaptcha-response' => Rule::requiredIf(plugin_active('Google reCaptcha')), // conditional reCAPTCHA
            new Recaptcha(),
            'i_agree' => ['required'],
        ]);

        $input = $request->all();

        $location = getLocation();
        $phone = $isPhone ? ($isCountry ? explode(':', $input['country'])[1] : $location->dial_code) . ' ' . $input['phone'] : $location->dial_code . ' ';
        $country = $isCountry ? explode(':', $input['country'])[0] : $location->name;

        $rank = Ranking::find(1);

        $user = User::create([
            'ranking_id' => $rank->id,
            'rankings' => json_encode([$rank->id]),
            'first_name' => $input['first_name'],
            'last_name' => $input['last_name'],
            'username' => $isUsername ? $input['username'] : $input['first_name'] . $input['last_name'] . rand(1000, 9999),
            'country' => $country,
            'phone' => $phone,
            'email' => $input['email'],
            'password' => Hash::make($input['password']),
        ]);

        // Add ranking bonus if applicable
        if ($rank->bonus > 0) {
            Txn::new($rank->bonus, 0, $rank->bonus, 'system', 'Ranking Bonus From ' . $rank->ranking, TxnType::Bonus, TxnStatus::Success, null, null, $user->id);
            $user->increment('profit_balance', $rank->bonus);
        }

        $shortcodes = [
            '[[full_name]]' => $input['first_name'] . ' ' . $input['last_name'],
            '[[message]]' => '.New User added our system.',
        ];

        // Notify admins and user
        $this->pushNotify('new_user', $shortcodes, route('admin.user.edit', $user->id), $user->id);
        $this->smsNotify('new_user', $shortcodes, $user->phone);

        // Referral event
        event(new UserReferred($request->cookie('invite'), $user));

        // Signup bonus
        if (setting('referral_signup_bonus', 'permission') && (float) setting('signup_bonus', 'fee') > 0) {
            $signupBonus = (float) setting('signup_bonus', 'fee');
            $user->increment('profit_balance', $signupBonus);
            Txn::new($signupBonus, 0, $signupBonus, 'system', 'Signup Bonus', TxnType::SignupBonus, TxnStatus::Success, null, null, $user->id);
            Session::put('signup_bonus', $signupBonus);
        }

        \Cookie::forget('invite');
        Auth::login($user);
        LoginActivities::add();

        return redirect(RouteServiceProvider::HOME);
    }
}
